﻿﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;

using System.Web.Mvc;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

using HIPS.Web.Components.ServiceModel;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.Test.Common;

using HIPS.Web.UI.Helpers;

using Moq;

namespace HIPS.Web.Test.PcehrView
{

    /// <summary>
    /// Unit tests for the PcehrAccessManager.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
    /// </history>
    [TestClass]
    public class PcehrAccessManagerTests : TestBase
    {

        #region Methods

        #region Tests

        #region Gain Access

        /// <summary>
        /// Test that the access manager behaves as expected when the PCEHR access status is "Access Granted".
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Access_Granted()
        {
            this.Access_Test(AccessCodeRequired.AccessGranted, false, true);
        }

        /// <summary>
        /// Test that the access manager behaves as expected when the PCEHR access status is "Without Code".
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Without_Code()
        {
            this.Access_Test(AccessCodeRequired.WithoutCode, false, true);
        }

        /// <summary>
        /// Test that the access manager behaves as expected when the PCEHR access status is "Unknown" and no code is required.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Unknown_Without_Code()
        {
            this.Access_Test(AccessCodeRequired.Unknown, true, true);
        }

        /// <summary>
        /// Test that the access manager behaves as expected when the PCEHR access status is "Unknown" and a code is required.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Unknown_With_Code()
        {
            this.Access_Test(AccessCodeRequired.Unknown, false, false);
        }

        /// <summary>
        /// Test that the access manager behaves as expected when the PCEHR access status is "With Code".
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void With_Code()
        {
            this.Access_Test(AccessCodeRequired.WithCode, false, false);
        }

        #endregion

        #endregion

        #region Helpers

        /// <summary>
        /// Execute a test against the PcehrAccessManager.
        /// </summary>
        /// <param name="accessLevel">Access level to the PCEHR for patients / documents.</param>
        /// <param name="allowUnknownAccessWithoutCode">For an Access Level of "Unknown", indicates whether access should be allowed without a code.</param>
        /// <param name="expectedResult">Boolean value providing expected result based on test parameters.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="13 January 2014">Initial version.</change>
        /// </history>
        private void Access_Test(AccessCodeRequired accessLevel, bool allowUnknownAccessWithoutCode, bool expectedResult)
        {
            //Arrange
            int numberOfHospitals = 5;
            var mockHospitalRepository = TestAssistant.MockRepository.GetHospitalRepository(numberOfHospitals);
            var selectedHospital = mockHospitalRepository.Object.GetHospitals(TestAssistant.HOSPITAL_CODE_SYSTEM)[2];
            var selectedHospitalId = selectedHospital.Codes.FirstOrDefault(c => c.CodeSystemCode == TestAssistant.HOSPITAL_CODE_SYSTEM).Code;
            var numberOfPatientsWithPcehr = 10;
            var numberOfPatientsWithoutPcehr = 5;
            var mockPatientRepository = TestAssistant.MockRepository.GetPatientRepository(selectedHospital, numberOfPatientsWithPcehr, numberOfPatientsWithoutPcehr);
            var selectedPatient = mockPatientRepository.Object.ListPatientsCurrentlyInHospital(TestAssistant.HOSPITAL_CODE_SYSTEM, selectedHospitalId, true, TestAssistant.GetCurrentUserDetails()).Data.PatientInHospitalList[2];
            var selectedPatientId = selectedPatient.Mrn;
            var mockPcehrViewRepository = TestAssistant.MockRepository.GetPcehrViewRepository(selectedHospital, selectedPatient, null, 0, accessLevel, allowUnknownAccessWithoutCode);
            var mockSettingsRepository = TestAssistant.MockRepository.GetSettingsRepository();

            //Act
            var pcehrAccessManager = new PcehrAccessManager(mockPatientRepository.Object, mockPcehrViewRepository.Object);
            var result = pcehrAccessManager.AllowAccess(new Mrn(selectedPatientId, selectedHospitalId, TestAssistant.HOSPITAL_CODE_SYSTEM), true, TestAssistant.GetCurrentUserDetails());

            //Assert
            Assert.IsNotNull(result, "Test for non-null result failed.");
            Assert.AreEqual(expectedResult, result, "Test for expected result failed.");
        }


        #endregion

        #endregion

    }

}
